
# ESP32 MQTT LED Control

This ESP32 project demonstrates how to control two LEDs via MQTT using the ESP-IDF framework:

- **LED1** (GPIO 26): Simple ON/OFF control
- **LED2** (GPIO 27): Brightness control using PWM

The project also saves and restores LED states using NVS (non-volatile storage).

---

## 🛠️ Features

- Connects ESP32 to a Wi-Fi network
- Subscribes to MQTT topics:
  - `esp32/led` → Controls LED1 ON/OFF
  - `esp32/led2/brightness` → Controls LED2 brightness (0–100%)
- Restores LED states on restart using NVS
- Debounces rapid LED2 updates using a 200ms interval

---

## 📦 Requirements

- ESP32 WROVER/DEV board
- [ESP-IDF](https://docs.espressif.com/projects/esp-idf/en/latest/esp32/get-started/) (v4.4 or later)
- MQTT Broker (e.g., [Mosquitto](https://mosquitto.org/))

---

## 🧑‍💻 How to Build and Flash the Code

### 1. Clone or Copy the Project

Place the code in an ESP-IDF project folder structure:

```
your_project/
├── main/
│   └── main.c
├── CMakeLists.txt
├── sdkconfig
└── ...
```

> ⚠️ Make sure you have initialized your ESP-IDF environment.

### 2. Configure Your Project

Run:

```bash
idf.py menuconfig
```

- Set target → `esp32`
- Ensure `Partition Table`, `NVS`, `WiFi`, and `MQTT` components are enabled
- Optionally: Enable logs under `Component config → Log output`

### 3. Build and Flash

```bash
idf.py build
idf.py -p /dev/ttyUSB0 flash monitor
```

Replace `/dev/ttyUSB0` with your correct COM port.

---

## 📡 MQTT Topics

| Topic                     | Payload        | Description                  |
|--------------------------|----------------|------------------------------|
| `esp32/led`              | `1` / `0`      | Turn LED1 ON or OFF          |
| `esp32/led2/brightness`  | `0` – `100`    | Set LED2 brightness percent  |

---

## 🌐 How to Enable Mosquitto MQTT Broker

### Install Mosquitto (on Ubuntu/Debian)

```bash
sudo apt update
sudo apt install mosquitto mosquitto-clients
sudo systemctl enable mosquitto
sudo systemctl start mosquitto
```

### Verify It’s Running

```bash
mosquitto -v
```

You should see something like:

```
mosquitto version 2.x starting
...
Listening on port 1883
```

### Test with MQTT CLI

Open two terminals:

**Terminal 1 – Subscriber:**

```bash
mosquitto_sub -h localhost -t "esp32/led"
```

**Terminal 2 – Publisher:**

```bash
mosquitto_pub -h localhost -t "esp32/led" -m "1"
```

---

## 📲 MQTT with Mobile App

You can use **IoT MQTT Panel** (Android/iOS) or **MQTT Explorer** (PC) to control LEDs.

---

## 📁 NVS Storage Keys

| Key              | Description            |
|------------------|------------------------|
| `led_state`      | LED1 ON/OFF (0 or 1)   |
| `led2_brightness`| LED2 brightness %      |

---

## 📌 Notes

- PWM for LED2 uses 10-bit resolution (0–1023)
- MQTT messages are throttled to one every 200ms for LED2 to avoid flickering
- You can configure your Wi-Fi and MQTT broker address by modifying:

```c
#define WIFI_SSID     "EV"
#define WIFI_PASSWORD "onlyforlubiev"
#define MQTT_BROKER   "mqtt://192.168.1.6"
```

---

## 🔧 Troubleshooting

- **ESP32 not connecting to Wi-Fi?**
  - Double-check SSID and password
  - Look at serial logs for errors

- **MQTT not working?**
  - Make sure broker is running and reachable
  - Verify broker IP and port match in code

---

## 📃 License

MIT License — free for personal and commercial use.


---

## 📱 Steps to Add a Slider Panel and Switch in IoT MQTT Panel (Mobile App)

Use the **IoT MQTT Panel** app to control your ESP32 via MQTT:

### 1. Launch the App and Set Up a Connection

- Open the **IoT MQTT Panel** app on your iOS or Android device.
- Tap **Setup a Connection**.

**Configure the connection:**

- **Connection Name**: e.g., `Home MQTT Broker`
- **Client ID**: e.g., `iOSClient1`
- **Broker Web/IP Address**: `192.168.1.6`
- **Port**: `1883` (or `8883` if using SSL)
- **Network Protocol**: `TCP`
- **SSL**: Optional (enable if your broker uses SSL)

➡️ Tap **Save** to store the connection.

---

### 2. Create a Dashboard

- Tap your connection name (e.g., `Home MQTT Broker`)
- Tap the **+ icon** to create a dashboard
- Name it something like: `ESP32 LEDs`
- Tap **Save**

---

### 3. Add a Slider Panel for LED2 (PWM Brightness Control)

Configure the slider as:

- **Panel Name**: `LED2 Brightness`
- **Publish Topic**: `esp32/led2/brightness`
- **Subscribe Topic**: *(Leave blank)*
- **Min Value**: `0`
- **Max Value**: `100`
- **Slider Step**: `1`
- **Unit**: `%`
- **Factor**: `1`
- **Decimal Precision**: `0`
- **Retain**: `Unchecked`
- **QoS**: `0`
- **Icon**: *(Optional: Light bulb icon)*
- **Panel Width**: `1x1` or larger for better usability

---

### 4. Add a Switch Panel for LED1 (GPIO 26 ON/OFF)

- Tap the **+ icon** again
- Choose **Switch** panel

Configure as:

- **Panel Name**: `LED1 Switch`
- **Publish Topic**: `esp32/led`
- **Subscribe Topic**: `esp32/led` *(optional)*
- **ON Payload**: `1`
- **OFF Payload**: `0`
- **Icon**: *(Optional: Switch/light icon)*

Tap **Save** to finish.

Now, you can remotely control LED1 and adjust LED2 brightness in real time!

---
