#include <stdio.h>
#include <string.h>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "esp_log.h"
#include "esp_wifi.h"
#include "esp_event.h"
#include "nvs_flash.h"
#include "mqtt_client.h"
#include "driver/gpio.h"
#include "driver/ledc.h"
#include <esp_timer.h>
/****************************************************************/
#define WIFI_SSID      "EV"      				/* Replace with your Wi-Fi SSID */
#define WIFI_PASSWORD  "onlyforlubiev"  		/* Replace with your Wi-Fi password */
#define MQTT_BROKER    "mqtt://192.168.1.6"
#define MQTT_TOPIC     "esp32/led"
#define LED_PIN        26
#define LED2_PIN       27
#define MQTT_TOPIC_LED2 "esp32/led2/brightness"
#define LEDC_CHANNEL_2 LEDC_CHANNEL_1
#define LEDC_TIMER_2   LEDC_TIMER_1
#define LEDC_FREQ_HZ   5000  					/* PWM frequency */
#define LEDC_RESOLUTION LEDC_TIMER_10_BIT  		/*10-bit resolution (0–1023)*/
#define STORAGE_NAMESPACE "storage"
/****************************************************************/
// RGB LED Pins
#define RGB_RED_PIN    18
#define RGB_GREEN_PIN  19
#define RGB_BLUE_PIN   21
#define LEDC_CHANNEL_RED   LEDC_CHANNEL_2
#define LEDC_CHANNEL_GREEN LEDC_CHANNEL_3
#define LEDC_CHANNEL_BLUE  LEDC_CHANNEL_4
#define LEDC_TIMER_RGB     LEDC_TIMER_2
#define MQTT_TOPIC_RGB_COLOR "esp32/rgb/color"
#define MQTT_TOPIC_RGB_BRIGHTNESS "esp32/rgb/brightness"
/****************************************************************/
void save_led_state(int state);
int load_led_state(void);
void save_led2_brightness(int brightness);
int load_led2_brightness(void);
static uint32_t last_update_time = 0;
static const char *TAG = "MQTT_LED";
/****************************************************************/
void save_rgb_color(int r, int g, int b);
void load_rgb_color(int *r, int *g, int *b);
void save_rgb_brightness(int brightness);
int load_rgb_brightness(void);
static uint32_t last_rgb_update_time = 0;
/****************************************************************/

static void wifi_event_handler(void *arg, esp_event_base_t event_base, int32_t event_id, void *event_data) 
{
    if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_STA_START) 
    {
        esp_wifi_connect();
    } 
    else if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_STA_DISCONNECTED) 
    {
        esp_wifi_connect();
        ESP_LOGI(TAG, "Retrying Wi-Fi connection...");
    } 
    else if (event_base == IP_EVENT && event_id == IP_EVENT_STA_GOT_IP) 
    {
        ESP_LOGI(TAG, "Wi-Fi connected, IP obtained");
    }
}

static void wifi_init(void) 
{
    esp_netif_init();
    esp_event_loop_create_default();
    esp_netif_create_default_wifi_sta();
    
    wifi_init_config_t cfg = WIFI_INIT_CONFIG_DEFAULT();
    esp_wifi_init(&cfg);

    esp_event_handler_register(WIFI_EVENT, ESP_EVENT_ANY_ID, &wifi_event_handler, NULL);
    esp_event_handler_register(IP_EVENT, IP_EVENT_STA_GOT_IP, &wifi_event_handler, NULL);

    wifi_config_t wifi_config = {
        .sta = {
            .ssid = WIFI_SSID,
            .password = WIFI_PASSWORD,
        },
    };
    esp_wifi_set_mode(WIFI_MODE_STA);
    esp_wifi_set_config(ESP_IF_WIFI_STA, &wifi_config);
    esp_wifi_start();
}

static void mqtt_event_handler(void *arg, esp_event_base_t event_base, int32_t event_id, void *event_data) 
{
    esp_mqtt_event_handle_t event = event_data;
    esp_mqtt_client_handle_t client = event->client;

    switch (event_id) 
    {
        case MQTT_EVENT_CONNECTED:
            ESP_LOGI(TAG, "MQTT connected to broker");
            ESP_LOGI(TAG, "Subscribing to %s", MQTT_TOPIC);
            int msg_id1 = esp_mqtt_client_subscribe(client, MQTT_TOPIC, 0);
            ESP_LOGI(TAG, "Subscription to %s, msg_id=%d", MQTT_TOPIC, msg_id1);
            ESP_LOGI(TAG, "Subscribing to %s", MQTT_TOPIC_LED2);
            int msg_id2 = esp_mqtt_client_subscribe(client, MQTT_TOPIC_LED2, 0);
            ESP_LOGI(TAG, "Subscription to %s, msg_id=%d", MQTT_TOPIC_LED2, msg_id2);
            /****************************************************************/
            int msg_id3 = esp_mqtt_client_subscribe(client, MQTT_TOPIC_RGB_COLOR, 0);
            ESP_LOGI(TAG, "Subscription to %s, msg_id=%d", MQTT_TOPIC_RGB_COLOR, msg_id3);
            int msg_id4 = esp_mqtt_client_subscribe(client, MQTT_TOPIC_RGB_BRIGHTNESS, 0);
            ESP_LOGI(TAG, "Subscription to %s, msg_id=%d", MQTT_TOPIC_RGB_BRIGHTNESS, msg_id4);
            /****************************************************************/
            break;
        case MQTT_EVENT_DISCONNECTED:
            ESP_LOGI(TAG, "MQTT disconnected from broker");
            break;
        case MQTT_EVENT_DATA:
    		ESP_LOGI(TAG, "MQTT data received, topic: %.*s, data: %.*s",event->topic_len, event->topic, event->data_len, event->data);
    		if (strncmp(event->topic, MQTT_TOPIC, event->topic_len) == 0) 
    		{
        		if (event->data_len > 0 && event->data[0] == '1') 
        		{
					gpio_set_level(LED_PIN, 1);
					save_led_state(1);  // Save to NVS
	    			ESP_LOGI(TAG, "LED turned ON");
        		} 
        		else if (event->data_len > 0 && event->data[0] == '0') 
        		{
            		gpio_set_level(LED_PIN, 0);
            		save_led_state(0);  // Save to NVS
            		ESP_LOGI(TAG, "LED turned OFF");
        		}
    		}
			else if (strncmp(event->topic, MQTT_TOPIC_LED2, event->topic_len) == 0) 
			{
				uint32_t current_time = esp_timer_get_time() / 1000; // Time in ms
                if (current_time - last_update_time < 200) 
                {
                    ESP_LOGI(TAG, "Ignoring rapid update to LED2");
                    break;
                }
                last_update_time = current_time;
            	char data_str[event->data_len + 1];
	       	 	strncpy(data_str, event->data, event->data_len);
	        	data_str[event->data_len] = '\0';
		        ESP_LOGI(TAG, "Received payload: %s", data_str);
		        int brightness = atoi(data_str);
		        ESP_LOGI(TAG, "Parsed brightness: %d", brightness);
		        if (brightness >= 0 && brightness <= 100) 
		        {
		            uint32_t duty = (brightness * 1023) / 100;
		            ESP_LOGI(TAG, "Setting duty: %d", duty);
		            ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_2, duty);
		            ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_2);
		            save_led2_brightness(brightness);
		            ESP_LOGI(TAG, "LED2 brightness set to %d%%", brightness);
	            	// Publish the brightness back to the topic
		            char brightness_str[16];
		            snprintf(brightness_str, sizeof(brightness_str), "%d", brightness);
		            esp_mqtt_client_publish(client, MQTT_TOPIC_LED2, brightness_str, 0, 0, 0);
                } 
            	else 
            	{
                		ESP_LOGE(TAG, "Invalid brightness value for LED2: %d", brightness);
            	}
        	}
        	/****************************************************************/
        	else if (strncmp(event->topic, MQTT_TOPIC_RGB_COLOR, event->topic_len) == 0)
        	{
				uint32_t current_time = esp_timer_get_time() / 1000;
				if (current_time - last_rgb_update_time < 200) 
				{
					ESP_LOGI(TAG, "Ignoring rapid update to RGB");
					break;
				}
				last_rgb_update_time = current_time;
				char data_str[event->data_len + 1];
				strncpy(data_str, event->data, event->data_len);
				data_str[event->data_len] = '\0';
				int r, g, b;
				if (sscanf(data_str, "%d,%d,%d", &r, &g, &b) == 3)
				{
					if (r >= 0 && r <= 255 && g >= 0 && g <= 255 && b >= 0 && b <= 255)
					{
						
						int brightness = load_rgb_brightness();
						uint32_t duty_r = ((r * 1023) / 255) * brightness / 100;
						uint32_t duty_g = ((g * 1023) / 255) * brightness / 100;
						uint32_t duty_b = ((b * 1023) / 255) * brightness / 100;
						ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_RED, duty_r);
						ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_GREEN, duty_g);
						ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_BLUE, duty_b);
						ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_RED);
						ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_GREEN);
						ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_BLUE);
						save_rgb_color(r, g, b);
						ESP_LOGI(TAG, "RGB set to R:%d, G:%d, B:%d with brightness %d%%",
						r, g, b, brightness);
						char color_str[32];
						snprintf(color_str, sizeof(color_str), "%d,%d,%d", r, g, b);
						esp_mqtt_client_publish(client, MQTT_TOPIC_RGB_COLOR, color_str, 0, 0, 0);
					}
					else
					{
						ESP_LOGE(TAG, "Invalid RGB values: R:%d, G:%d, B:%d", r, g, b);
					}
				}
				else
				{
					ESP_LOGE(TAG, "Invalid RGB format: %s", data_str);
				}
			}
			else if (strncmp(event->topic, MQTT_TOPIC_RGB_BRIGHTNESS, event->topic_len) == 0)
			{
				uint32_t current_time = esp_timer_get_time() / 1000;
                if (current_time - last_rgb_update_time < 200) 
                {
                    ESP_LOGI(TAG, "Ignoring rapid update to RGB brightness");
                    break;
                }
                last_rgb_update_time = current_time;
                char data_str[event->data_len + 1];
                strncpy(data_str, event->data, event->data_len);
                data_str[event->data_len] = '\0';
                int brightness = atoi(data_str);
				if (brightness >= 0 && brightness <= 100)
				{
					int r, g, b;
                    load_rgb_color(&r, &g, &b);
                    uint32_t duty_r = ((r * 1023) / 255) * brightness / 100;
                    uint32_t duty_g = ((g * 1023) / 255) * brightness / 100;
                    uint32_t duty_b = ((b * 1023) / 255) * brightness / 100;
                    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_RED, duty_r);
                    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_GREEN, duty_g);
                    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_BLUE, duty_b);
                    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_RED);
                    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_GREEN);
                    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_BLUE);
                    save_rgb_brightness(brightness);
                    ESP_LOGI(TAG, "RGB brightness set to %d%%", brightness);
                    char brightness_str[16];
                    snprintf(brightness_str, sizeof(brightness_str), "%d", brightness);
                    esp_mqtt_client_publish(client, MQTT_TOPIC_RGB_BRIGHTNESS, brightness_str, 0, 0, 0);
				}
				else
				{
					ESP_LOGE(TAG, "Invalid RGB brightness value: %d", brightness);
				}
				/****************************************************************/
			}
        	else 
        	{
				ESP_LOGI(TAG, "Unknown topic received: %.*s", event->topic_len, event->topic);
			}
		break;
    default:
        ESP_LOGI(TAG, "Other MQTT event id: %d", event_id);
        break;
    }
}

static void mqtt_init(void) 
{
    esp_mqtt_client_config_t mqtt_cfg = {
        .broker = {
            .address = {
                .uri = MQTT_BROKER,
                .port = 1883,
            },
        },
    };
    esp_mqtt_client_handle_t client = esp_mqtt_client_init(&mqtt_cfg);
    esp_mqtt_client_register_event(client, ESP_EVENT_ANY_ID, mqtt_event_handler, NULL);
    esp_mqtt_client_start(client);
}

void save_led_state(int state) 
{
    nvs_handle_t nvs_handle;
    esp_err_t err = nvs_open(STORAGE_NAMESPACE, NVS_READWRITE, &nvs_handle);
    if (err == ESP_OK) 
    {
        nvs_set_i32(nvs_handle, "led_state", state);
        nvs_commit(nvs_handle);
        nvs_close(nvs_handle);
    }
}

int load_led_state(void) 
{
    nvs_handle_t nvs_handle;
    int32_t state = 0; // Default OFF
    esp_err_t err = nvs_open(STORAGE_NAMESPACE, NVS_READONLY, &nvs_handle);
    if (err == ESP_OK) 
    {
        nvs_get_i32(nvs_handle, "led_state", &state);
        nvs_close(nvs_handle);
    }
    return state;
}

void save_led2_brightness(int brightness) 
{
    nvs_handle_t nvs_handle;
    esp_err_t err = nvs_open(STORAGE_NAMESPACE, NVS_READWRITE, &nvs_handle);
    if (err == ESP_OK) {
        nvs_set_i32(nvs_handle, "led2_brightness", brightness);
        nvs_commit(nvs_handle);
        nvs_close(nvs_handle);
    }
}

int load_led2_brightness(void) 
{
    nvs_handle_t nvs_handle;
    int32_t brightness = 0;
    esp_err_t err = nvs_open(STORAGE_NAMESPACE, NVS_READONLY, &nvs_handle);
    if (err == ESP_OK) {
        nvs_get_i32(nvs_handle, "led2_brightness", &brightness);
        nvs_close(nvs_handle);
    }
    return brightness;
}

/****************************************************************/
void save_rgb_color(int r, int g, int b) 
{
    nvs_handle_t nvs_handle;
    esp_err_t err = nvs_open(STORAGE_NAMESPACE, NVS_READWRITE, &nvs_handle);
    if (err == ESP_OK) 
    {
        nvs_set_i32(nvs_handle, "rgb_r", r);
        nvs_set_i32(nvs_handle, "rgb_g", g);
        nvs_set_i32(nvs_handle, "rgb_b", b);
        nvs_commit(nvs_handle);
        nvs_close(nvs_handle);
    }
}

void load_rgb_color(int *r, int *g, int *b) 
{
    nvs_handle_t nvs_handle;
    int32_t r_val = 255, g_val = 255, b_val = 255; // Default to white
    esp_err_t err = nvs_open(STORAGE_NAMESPACE, NVS_READONLY, &nvs_handle);
    if (err == ESP_OK) 
    {
        nvs_get_i32(nvs_handle, "rgb_r", &r_val);
        nvs_get_i32(nvs_handle, "rgb_g", &g_val);
        nvs_get_i32(nvs_handle, "rgb_b", &b_val);
        nvs_close(nvs_handle);
    }
    *r = r_val;
    *g = g_val;
    *b = b_val;
}

void save_rgb_brightness(int brightness) 
{
    nvs_handle_t nvs_handle;
    esp_err_t err = nvs_open(STORAGE_NAMESPACE, NVS_READWRITE, &nvs_handle);
    if (err == ESP_OK) 
    {
        nvs_set_i32(nvs_handle, "rgb_brightness", brightness);
        nvs_commit(nvs_handle);
        nvs_close(nvs_handle);
    }
}

int load_rgb_brightness(void) 
{
    nvs_handle_t nvs_handle;
    int32_t brightness = 100; // Default to full brightness
    esp_err_t err = nvs_open(STORAGE_NAMESPACE, NVS_READONLY, &nvs_handle);
    if (err == ESP_OK) 
    {
        nvs_get_i32(nvs_handle, "rgb_brightness", &brightness);
        nvs_close(nvs_handle);
    }
    return brightness;
}
/****************************************************************/


static void led_init(void) 
{
    // Initialize LED1 (ON/OFF)
    gpio_config_t io_conf = {
        .pin_bit_mask = (1ULL << LED_PIN),
        .mode = GPIO_MODE_OUTPUT,
        .pull_up_en = 0,
        .pull_down_en = 0,
        .intr_type = GPIO_INTR_DISABLE,
    };
    gpio_config(&io_conf);
    int last_state = load_led_state();
    gpio_set_level(LED_PIN, last_state);
    ESP_LOGI(TAG, "LED1 restored to state: %d", last_state);

    // Initialize LED2 (PWM)
    ledc_timer_config_t ledc_timer = {
        .speed_mode = LEDC_HIGH_SPEED_MODE,
        .timer_num = LEDC_TIMER_2,
        .duty_resolution = LEDC_RESOLUTION,
        .freq_hz = LEDC_FREQ_HZ,
        .clk_cfg = LEDC_AUTO_CLK,
    };
    ledc_timer_config(&ledc_timer);
    ledc_channel_config_t ledc_channel = {
        .gpio_num = LED2_PIN,
        .speed_mode = LEDC_HIGH_SPEED_MODE,
        .channel = LEDC_CHANNEL_2,
        .intr_type = LEDC_INTR_DISABLE,
        .timer_sel = LEDC_TIMER_2,
        .duty = 0,
        .hpoint = 0,
    };
    ledc_channel_config(&ledc_channel);
    int brightness = load_led2_brightness();
    uint32_t duty = (brightness * 1023) / 100;
    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_2, duty);
    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_2);
    ESP_LOGI(TAG, "LED2 restored to brightness: %d%%", brightness);
    
    /****************************************************************/
    // Initialize RGB LED (PWM for each channel)
    ledc_channel_config_t ledc_red_channel = {
        .gpio_num = RGB_RED_PIN,
        .speed_mode = LEDC_HIGH_SPEED_MODE,
        .channel = LEDC_CHANNEL_RED,
        .intr_type = LEDC_INTR_DISABLE,
        .timer_sel = LEDC_TIMER_RGB,
        .duty = 0,
        .hpoint = 0,
    };
    ledc_channel_config_t ledc_green_channel = {
        .gpio_num = RGB_GREEN_PIN,
        .speed_mode = LEDC_HIGH_SPEED_MODE,
        .channel = LEDC_CHANNEL_GREEN,
        .intr_type = LEDC_INTR_DISABLE,
        .timer_sel = LEDC_TIMER_RGB,
        .duty = 0,
        .hpoint = 0,
    };
    ledc_channel_config_t ledc_blue_channel = {
        .gpio_num = RGB_BLUE_PIN,
        .speed_mode = LEDC_HIGH_SPEED_MODE,
        .channel = LEDC_CHANNEL_BLUE,
        .intr_type = LEDC_INTR_DISABLE,
        .timer_sel = LEDC_TIMER_RGB,
        .duty = 0,
        .hpoint = 0,
    };
    ledc_timer_config(&ledc_timer); // Reuse same timer config for RGB
    ledc_channel_config(&ledc_red_channel);
    ledc_channel_config(&ledc_green_channel);
    ledc_channel_config(&ledc_blue_channel);
    int r, g, b;
    load_rgb_color(&r, &g, &b);
    brightness = load_rgb_brightness();
    uint32_t duty_r = ((r * 1023) / 255) * brightness / 100;
    uint32_t duty_g = ((g * 1023) / 255) * brightness / 100;
    uint32_t duty_b = ((b * 1023) / 255) * brightness / 100;
    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_RED, duty_r);
    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_GREEN, duty_g);
    ledc_set_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_BLUE, duty_b);
    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_RED);
    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_GREEN);
    ledc_update_duty(LEDC_HIGH_SPEED_MODE, LEDC_CHANNEL_BLUE);
    ESP_LOGI(TAG, "RGB LED restored to R:%d, G:%d, B:%d, brightness: %d%%", r, g, b, brightness);
}

void app_main(void) 
{
    ESP_LOGI(TAG, "Starting MQTT LED Control");
    
    // Initialize NVS
    esp_err_t ret = nvs_flash_init();
    if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) 
    {
        nvs_flash_erase();
        ret = nvs_flash_init();
    }
    ESP_ERROR_CHECK(ret);

    led_init();
    wifi_init();
    mqtt_init();
}
