
# ESP32 MQTT LED Control

This ESP32 project demonstrates how to control two LEDs via MQTT using the ESP-IDF framework:

- **LED1** (GPIO 26): Simple ON/OFF control
- **LED2** (GPIO 27): Brightness control using PWM

The project also saves and restores LED states using NVS (non-volatile storage).

---

## 🛠️ Features

- Connects ESP32 to a Wi-Fi network
- Subscribes to MQTT topics:
  - `esp32/led` → Controls LED1 ON/OFF
  - `esp32/led2/brightness` → Controls LED2 brightness (0–100%)
- Restores LED states on restart using NVS
- Debounces rapid LED2 updates using a 200ms interval

---

## 📦 Requirements

- ESP32 WROVER/DEV board
- [ESP-IDF](https://docs.espressif.com/projects/esp-idf/en/latest/esp32/get-started/) (v4.4 or later)
- MQTT Broker (e.g., [Mosquitto](https://mosquitto.org/))

---

## 🧑‍💻 How to Build and Flash the Code

### 1. Clone or Copy the Project

Place the code in an ESP-IDF project folder structure:

```
your_project/
├── main/
│   └── main.c
├── CMakeLists.txt
├── sdkconfig
└── ...
```

> ⚠️ Make sure you have initialized your ESP-IDF environment.

### 2. Configure Your Project

Run:

```bash
idf.py menuconfig
```

- Set target → `esp32`
- Ensure `Partition Table`, `NVS`, `WiFi`, and `MQTT` components are enabled
- Optionally: Enable logs under `Component config → Log output`

### 3. Build and Flash

```bash
idf.py build
idf.py -p /dev/ttyUSB0 flash monitor
```

Replace `/dev/ttyUSB0` with your correct COM port.

---

## 📡 MQTT Topics

| Topic                     | Payload        | Description                  |
|--------------------------|----------------|------------------------------|
| `esp32/led`              | `1` / `0`      | Turn LED1 ON or OFF          |
| `esp32/led2/brightness`  | `0` – `100`    | Set LED2 brightness percent  |

---

## 🌐 How to Enable Mosquitto MQTT Broker

### Install Mosquitto (on Ubuntu/Debian)

```bash
sudo apt update
sudo apt install mosquitto mosquitto-clients
sudo systemctl enable mosquitto
sudo systemctl start mosquitto
```

### Verify It’s Running

```bash
mosquitto -v
```

You should see something like:

```
mosquitto version 2.x starting
...
Listening on port 1883
```

### Test with MQTT CLI

Open two terminals:

**Terminal 1 – Subscriber:**

```bash
mosquitto_sub -h localhost -t "esp32/led"
```

**Terminal 2 – Publisher:**

```bash
mosquitto_pub -h localhost -t "esp32/led" -m "1"
```

---

## 📲 MQTT with Mobile App

You can use **IoT MQTT Panel** (Android/iOS) or **MQTT Explorer** (PC) to control LEDs.

---

## 📁 NVS Storage Keys

| Key              | Description            |
|------------------|------------------------|
| `led_state`      | LED1 ON/OFF (0 or 1)   |
| `led2_brightness`| LED2 brightness %      |

---

## 📌 Notes

- PWM for LED2 uses 10-bit resolution (0–1023)
- MQTT messages are throttled to one every 200ms for LED2 to avoid flickering
- You can configure your Wi-Fi and MQTT broker address by modifying:

```c
#define WIFI_SSID     "EV"
#define WIFI_PASSWORD "onlyforlubiev"
#define MQTT_BROKER   "mqtt://192.168.1.6"
```

---

## 🔧 Troubleshooting

- **ESP32 not connecting to Wi-Fi?**
  - Double-check SSID and password
  - Look at serial logs for errors

- **MQTT not working?**
  - Make sure broker is running and reachable
  - Verify broker IP and port match in code

---

## 📃 License

MIT License — free for personal and commercial use.


---

## 📱 Steps to Add a Slider Panel and Switch in IoT MQTT Panel (Mobile App)

Use the **IoT MQTT Panel** app to control your ESP32 via MQTT:

### 1. Launch the App and Set Up a Connection

- Open the **IoT MQTT Panel** app on your iOS or Android device.
- Tap **Setup a Connection**.

**Configure the connection:**

- **Connection Name**: e.g., `Home MQTT Broker`
- **Client ID**: e.g., `iOSClient1`
- **Broker Web/IP Address**: `192.168.1.6`
- **Port**: `1883` (or `8883` if using SSL)
- **Network Protocol**: `TCP`
- **SSL**: Optional (enable if your broker uses SSL)

➡️ Tap **Save** to store the connection.

---

### 2. Create a Dashboard

- Tap your connection name (e.g., `Home MQTT Broker`)
- Tap the **+ icon** to create a dashboard
- Name it something like: `ESP32 LEDs`
- Tap **Save**

---

### 3. Add a Slider Panel for LED2 (PWM Brightness Control)

Configure the slider as:

- **Panel Name**: `LED2 Brightness`
- **Publish Topic**: `esp32/led2/brightness`
- **Subscribe Topic**: *(Leave blank)*
- **Min Value**: `0`
- **Max Value**: `100`
- **Slider Step**: `1`
- **Unit**: `%`
- **Factor**: `1`
- **Decimal Precision**: `0`
- **Retain**: `Unchecked`
- **QoS**: `0`
- **Icon**: *(Optional: Light bulb icon)*
- **Panel Width**: `1x1` or larger for better usability

---

### 4. Add a Switch Panel for LED1 (GPIO 26 ON/OFF)

- Tap the **+ icon** again
- Choose **Switch** panel

Configure as:

- **Panel Name**: `LED1 Switch`
- **Publish Topic**: `esp32/led`
- **Subscribe Topic**: `esp32/led` *(optional)*
- **ON Payload**: `1`
- **OFF Payload**: `0`
- **Icon**: *(Optional: Switch/light icon)*

Tap **Save** to finish.

Now, you can remotely control LED1 and adjust LED2 brightness in real time!

---

###5. 3. Add Panels for RGB Color and Brightness

- The app supports various panels, including Color Picker for RGB values and Slider for brightness. You’ll create two panels to control the esp32/rgb/color and esp32/rgb/brightness topics.

- Panel 1: Color Picker for RGB Color
Add Color Picker Panel:
In the "RGB LED Control" dashboard, tap the + icon to add a panel.
Select Color Picker from the panel types.
Configure the panel:
Name: Enter "RGB Color".
Topic: Enter esp32/rgb/color (matches the topic in your code).
Publish: Enable (the app will publish color values to this topic).
Subscribe: Enable (to receive and display the current RGB color published by the ESP32).
Payload Format: The Color Picker sends RGB values in a comma-separated format (e.g., "255,0,0"). Your code expects this format (R,G,B), so no additional JSON configuration is needed.
Icon: Choose an icon (e.g., a light bulb) from the 250+ available icons for visual appeal.
Retain: Enable if you want the broker to retain the last color message (useful for restoring state).
Tap Save.
How It Works:
When you select a color in the Color Picker, the app publishes the RGB values (e.g., "255,128,0" for orange) to esp32/rgb/color.
The ESP32 receives this, sets the PWM duties for the RGB LED (scaled by the current brightness), and publishes the color back to the topic for feedback.
The Color Picker updates to reflect the current color if the ESP32 publishes to esp32/rgb/color.
Panel 2: Slider for Brightness
Add Slider Panel:
In the same dashboard, tap the + icon again.
Select Slider from the panel types.
Configure the panel:
Name: Enter "RGB Brightness".
Topic: Enter esp32/rgb/brightness (matches the topic in your code).
Publish: Enable (to send brightness values).
Subscribe: Enable (to display the current brightness).
Min Value: Set to 0.
Max Value: Set to 100 (your code expects brightness as a percentage, 0-100).
Step: Set to 1 for fine-grained control.
Payload Format: The Slider sends numeric values (e.g., "50" for 50%). Your code parses this with atoi, so no additional formatting is needed.
Icon: Choose an icon (e.g., a brightness symbol).
Retain: Enable if you want the broker to retain the last brightness value.
Tap Save.
How It Works:
Moving the slider publishes a value (e.g., "75" for 75%) to esp32/rgb/brightness.
The ESP32 adjusts the PWM duties for all RGB channels based on this brightness and the current color, then publishes the brightness back to the topic.
The Slider updates to reflect the current brightness value.
