#include "stm32g0xx_hal.h"

// Define CS pin - change according to your hardware
#define CS5460A_CS_PIN      GPIO_PIN_0
#define CS5460A_CS_PORT     GPIOA

// SPI handle - configure this in your STM32CubeIDE or similar
extern SPI_HandleTypeDef hspi1;

// CS5460A Command Definitions
#define CS5460A_CMD_WRITE   0x40
#define CS5460A_CMD_READ    0x20
#define CS5460A_CMD_SYNC0   0xFE
#define CS5460A_CMD_SYNC1   0xFF

// Important CS5460A Registers
#define CS5460A_REG_CONFIG  0x00
#define CS5460A_REG_CURRENT 0x01
#define CS5460A_REG_VOLTAGE 0x02
#define CS5460A_REG_POWER   0x03
#define CS5460A_REG_ENERGY  0x04

// Function prototypes
void CS5460A_Init(void);
void CS5460A_WriteReg(uint8_t reg, uint32_t value);
uint32_t CS5460A_ReadReg(uint8_t reg);
void CS5460A_Reset(void);
float CS5460A_ReadCurrent(void);
float CS5460A_ReadVoltage(void);
float CS5460A_ReadPower(void);

void CS5460A_CS_Enable(void) {
    HAL_GPIO_WritePin(CS5460A_CS_PORT, CS5460A_CS_PIN, GPIO_PIN_RESET);
}

void CS5460A_CS_Disable(void) {
    HAL_GPIO_WritePin(CS5460A_CS_PORT, CS5460A_CS_PIN, GPIO_PIN_SET);
}

// Initialize the CS5460A
void CS5460A_Init(void) {
    // Initialize CS pin
    GPIO_InitTypeDef GPIO_InitStruct = {0};
    GPIO_InitStruct.Pin = CS5460A_CS_PIN;
    GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
    GPIO_InitStruct.Pull = GPIO_NOPULL;
    GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_HIGH;
    HAL_GPIO_Init(CS5460A_CS_PORT, &GPIO_InitStruct);
    
    CS5460A_CS_Disable();
    
    // Reset the CS5460A
    CS5460A_Reset();
    
    // Wait for initialization (adjust based on your clock)
    HAL_Delay(10);
    
    // Configure the CS5460A (example configuration)
    // Configure for continuous conversion mode, current and voltage channels enabled
    CS5460A_WriteReg(CS5460A_REG_CONFIG, 0x000000);
    
    // Additional configuration as needed...
}

// Write to a CS5460A register
void CS5460A_WriteReg(uint8_t reg, uint32_t value) {
    uint8_t txData[4];
    
    txData[0] = CS5460A_CMD_WRITE | (reg & 0x1F);
    txData[1] = (value >> 16) & 0xFF;
    txData[2] = (value >> 8) & 0xFF;
    txData[3] = value & 0xFF;
    
    CS5460A_CS_Enable();
    HAL_SPI_Transmit(&hspi1, txData, 4, HAL_MAX_DELAY);
    CS5460A_CS_Disable();
}

// Read from a CS5460A register
uint32_t CS5460A_ReadReg(uint8_t reg) {
    uint8_t txData[4] = {0};
    uint8_t rxData[4] = {0};
    
    txData[0] = CS5460A_CMD_READ | (reg & 0x1F);
    
    CS5460A_CS_Enable();
    HAL_SPI_TransmitReceive(&hspi1, txData, rxData, 4, HAL_MAX_DELAY);
    CS5460A_CS_Disable();
    
    return ((uint32_t)rxData[1] << 16) | ((uint32_t)rxData[2] << 8) | rxData[3];
}

// Reset the CS5460A
void CS5460A_Reset(void) {
    uint8_t sync0 = CS5460A_CMD_SYNC0;
    uint8_t sync1 = CS5460A_CMD_SYNC1;
    
    CS5460A_CS_Enable();
    HAL_SPI_Transmit(&hspi1, &sync0, 1, HAL_MAX_DELAY);
    HAL_SPI_Transmit(&hspi1, &sync1, 1, HAL_MAX_DELAY);
    CS5460A_CS_Disable();
    
    HAL_Delay(1); // Wait for reset to complete
}

// Read current value (in Amperes)
float CS5460A_ReadCurrent(void) {
    uint32_t rawValue = CS5460A_ReadReg(CS5460A_REG_CURRENT);
    int32_t signedValue = (int32_t)rawValue;
    
    // Convert to float (adjust scaling based on your current sensor)
    // Example: 250mV full scale for current channel
    return (float)signedValue * 0.000038147; // 250mV / 2^23
}

// Read voltage value (in Volts)
float CS5460A_ReadVoltage(void) {
    uint32_t rawValue = CS5460A_ReadReg(CS5460A_REG_VOLTAGE);
    int32_t signedValue = (int32_t)rawValue;
    
    // Convert to float (adjust scaling based on your voltage divider)
    // Example: 250mV full scale for voltage channel
    return (float)signedValue * 0.000038147; // 250mV / 2^23
}

// Read power value (in Watts)
float CS5460A_ReadPower(void) {
    uint32_t rawValue = CS5460A_ReadReg(CS5460A_REG_POWER);
    int32_t signedValue = (int32_t)rawValue;
    
    // Convert to float (adjust scaling based on your configuration)
    return (float)signedValue * 0.000076294; // (250mV * 250mV) / (2^23 * R)
}

int main(void) {
    // HAL initialization (generated by STM32CubeIDE)
    HAL_Init();
    SystemClock_Config();
    MX_GPIO_Init();
    MX_SPI1_Init();
    
    // Initialize CS5460A
    CS5460A_Init();
    
    while (1) {
        // Read and process measurements
        float current = CS5460A_ReadCurrent();
        float voltage = CS5460A_ReadVoltage();
        float power = CS5460A_ReadPower();
        
        // Here you can display the values or send them via UART, etc.
        // For example:
        printf("Current: %.3f A, Voltage: %.1f V, Power: %.1f W\n", current, voltage, power);
        
        HAL_Delay(1000); // Read every second
    }
}